<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\ArchivesExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\TherapistArchives\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Imports\SuperAdmin\ArchivesImport;
use App\Models\Archive;
use App\Models\Therapist;
use App\Models\ArchiveCategory;
use App\Models\Tag;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;

class TherapistArchivesController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist.add_archive');
        $this->middleware('permission:therapist.add_archive', ['only' => ['store']]);
        $this->middleware('permission:therapist.add_archive', ['only' => ['update']]);
        $this->middleware('permission:therapist.add_archive', ['only' => ['destroy']]);
        $this->middleware('permission:therapist.add_archive', ['only' => ['export']]);
        $this->middleware('permission:therapist.add_archive', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null, $therapist)
    {
        if ($req != null) {
            $therapist_archives =  $therapist->therapist_archives();
            if ($req->trash && $req->trash == 'with') {
                $therapist_archives =  $therapist_archives->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_archives =  $therapist_archives->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_archives = $therapist_archives->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_archives = $therapist_archives->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapist_archives = $therapist_archives->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapist_archives = $therapist_archives->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_archives = $therapist_archives->get();
                return $therapist_archives;
            }
            $therapist_archives = $therapist_archives->get();
            return $therapist_archives;
        }
        $therapist_archives = $therapist->therapist_archives()->withAll()->orderBy('id', 'desc')->get();
        return $therapist_archives;
    }


    /*********View All TherapistArchives  ***********/
    public function index(Request $request, Therapist $therapist)
    {
        $therapist_archives = $this->getter($request, null, $therapist);
        return view('super_admins.therapists.therapist_archives.index', compact('therapist_archives', 'therapist'));
    }

    /*********View Create Form of Archive  ***********/
    public function create(Therapist $therapist)
    {
        $archive_categories = ArchiveCategory::active()->get();
        $tags = Tag::active()->get();

        return view('super_admins.therapists.therapist_archives.create', compact('archive_categories', 'therapist', 'tags'));
    }

    /*********Store Archive  ***********/
    public function store(CreateRequest $request, Therapist $therapist)
    {

        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $request->merge(['created_by_user_id' => auth()->user()->id]);
            $data = $request->all();
            $data['image'] = uploadCroppedFile($request, 'image', 'therapist_archives');
            $therapist_archive = $therapist->therapist_archives()->create($data);
            $therapist_archive->slug = Str::slug($therapist_archive->name . ' ' . $therapist_archive->id, '-');
            $therapist_archive->save();
            $therapist_archive = $therapist->therapist_archives()->withAll()->find($therapist_archive->id);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_archives.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_archives.index', $therapist->id)->with('message', 'Archive Created Successfully')->with('message_type', 'success');
    }

    /*********View Archive  ***********/
    public function show(Therapist $therapist, Archive $therapist_archive)
    {
        if ($therapist->id != $therapist_archive->therapist_id) {
            return redirect()->back()->with('message', 'TherapistArchive Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapists.therapist_archives.show', compact('therapist_archive', 'therapist'));
    }

    /*********View Edit Form of Archive  ***********/
    public function edit(Therapist $therapist, Archive $therapist_archive)
    {
        if ($therapist->id != $therapist_archive->therapist_id) {
            return redirect()->back()->with('message', 'TherapistArchive Not Found')->with('message_type', 'error');
        }
        $archive_categories = ArchiveCategory::active()->get();
        $tags = Tag::active()->get();
        return view('super_admins.therapists.therapist_archives.edit', compact('therapist_archive', 'archive_categories', 'therapist', 'tags'));
    }

    /*********Update Archive  ***********/
    public function update(CreateRequest $request, Therapist $therapist, Archive $therapist_archive)
    {
        if ($therapist->id != $therapist_archive->therapist_id) {
            return redirect()->back()->with('message', 'TherapistArchive Not Found')->with('message_type', 'error');
        }
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $request->merge(['last_updated_by_user_id' => auth()->user()->id]);
            $data = $request->all();
            if ($request->image) {
                $data['image'] = uploadCroppedFile($request, 'image', 'therapist_archives', $therapist_archive->image);
            } else {
                $data['image'] = $therapist_archive->image;
            }
            $therapist_archive->update($data);
            $therapist_archive = Archive::find($therapist_archive->id);
            $slug = Str::slug($therapist_archive->name . ' ' . $therapist_archive->id, '-');
            $therapist_archive->update([
                'slug' => $slug
            ]);
            $therapist_archive->tags()->sync($request->tag_ids);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_archives.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_archives.index', $therapist->id)->with('message', 'Archive Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapist_archives = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapist_archives." . $extension;
        return Excel::download(new ArchivesExport($therapist_archives), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new ArchivesImport, $file);
        return redirect()->back()->with('message', 'Archive Categories imported Successfully')->with('message_type', 'success');
    }


    /*********Soft DELETE Archive ***********/
    public function destroy(Therapist $therapist, Archive $therapist_archive)
    {
        if ($therapist->id != $therapist_archive->therapist_id) {
            return redirect()->back()->with('message', 'TherapistArchive Not Found')->with('message_type', 'error');
        }
        $therapist_archive->delete();
        return redirect()->back()->with('message', 'Archive Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE Archive ***********/
    public function destroyPermanently(Request $request, Therapist $therapist, $therapist_archive)
    {
        if ($therapist->id != $therapist_archive->therapist_id) {
            return redirect()->back()->with('message', 'TherapistArchive Not Found')->with('message_type', 'error');
        }
        $therapist_archive = Archive::withTrashed()->find($therapist_archive);
        if ($therapist_archive) {
            if ($therapist_archive->trashed()) {
                if ($therapist_archive->image && file_exists(public_path($therapist_archive->image))) {
                    unlink(public_path($therapist_archive->image));
                }
                $therapist_archive->forceDelete();
                return redirect()->back()->with('message', 'Archive Category Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'Archive Category is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'Archive Category Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore Archive***********/
    public function restore(Request $request, Therapist $therapist, $therapist_archive)
    {
        if ($therapist->id != $therapist_archive->therapist_id) {
            return redirect()->back()->with('message', 'TherapistArchive Not Found')->with('message_type', 'error');
        }
        $therapist_archive = Archive::withTrashed()->find($therapist_archive);
        if ($therapist_archive->trashed()) {
            $therapist_archive->restore();
            return redirect()->back()->with('message', 'Archive Category Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'Archive Category Not Found')->with('message_type', 'error');
        }
    }
}
